<?php

/**
 * Theme functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Travel_Gem
 */

if (! function_exists('travel_gem_setup')) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function travel_gem_setup()
	{
		/*
		 * Make theme available for translation.
		 */
		load_theme_textdomain('travel-gem', get_template_directory() . '/languages');

		// Add default posts and comments RSS feed links to head.
		add_theme_support('automatic-feed-links');

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support('title-tag');

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support('post-thumbnails');
		add_image_size('travel-gem-trip-thumb', 400, 500, true);

		// This theme uses wp_nav_menu() in two locations.
		register_nav_menus(array(
			'menu-1'      => esc_html__('Primary', 'travel-gem'),
			'menu-footer' => esc_html__('Footer', 'travel-gem'),
		));

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support('html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		));

		// Add theme support for selective refresh for widgets.
		add_theme_support('customize-selective-refresh-widgets');

		// Load default block styles.
		add_theme_support('wp-block-styles');

		// Add support for responsive embeds.
		add_theme_support('responsive-embeds');

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support('custom-logo', array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		));

		// Add support for Custom Header.
		add_theme_support('custom-header', apply_filters('travel_gem_custom_header_args', array(
			'default-image' => get_template_directory_uri() . '/images/default-banner.jpg',
			'width'         => 1920,
			'height'        => 500,
			'flex-height'   => true,
			'header-text'   => false,
		)));

		// Register default headers.
		register_default_headers(array(
			'default-banner' => array(
				'url'           => '%s/images/default-banner.jpg',
				'thumbnail_url' => '%s/images/default-banner.jpg',
				'description'   => esc_html_x('Default Banner', 'header image description', 'travel-gem'),
			),

		));

		// Load Supports.
		require trailingslashit(get_template_directory()) . 'inc/support.php';
	}

endif;

add_action('after_setup_theme', 'travel_gem_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function travel_gem_content_width()
{
	$GLOBALS['content_width'] = apply_filters('travel_gem_content_width', 811);
}
add_action('after_setup_theme', 'travel_gem_content_width', 0);

if (! function_exists('travel_gem_template_redirect')) :
	/**
	 * Set the content width in pixels, based on the theme's design and stylesheet for different value other than the default one
	 *
	 * @global int $content_width
	 */
	function travel_gem_template_redirect()
	{
		$global_layout = travel_gem_get_option('global_layout');

		if ('no-sidebar' === $global_layout) {
			$GLOBALS['content_width'] = 1230;
		}

		// Three Columns
		elseif ('three-columns' == $global_layout) {
			$GLOBALS['content_width'] = 565;
		}
	}
endif;
add_action('template_redirect', 'travel_gem_template_redirect');

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function travel_gem_widgets_init()
{

	register_sidebar(array(
		'name'          => esc_html__('Primary Sidebar', 'travel-gem'),
		'id'            => 'sidebar-1',
		'description'   => esc_html__('Add widgets here to appear in your primary sidebar.', 'travel-gem'),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	));
	register_sidebar(array(
		'name'          => esc_html__('Secondary Sidebar', 'travel-gem'),
		'id'            => 'sidebar-2',
		'description'   => esc_html__('Add widgets here to appear in your secondary sidebar.', 'travel-gem'),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	));

	for ($i = 1; $i <= 4; $i++) {
		register_sidebar(array(
			'name'          => sprintf(esc_html__('Footer %d', 'travel-gem'), $i),
			'id'            => 'footer-' . $i,
			'before_widget' => '<div id="%1$s" class="widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		));
	}
}

add_action('widgets_init', 'travel_gem_widgets_init');

/**
 * Enqueue scripts and styles.
 */
function travel_gem_scripts()
{
	$min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

	// Font Awesome
	wp_enqueue_style('travel-gem-font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/font-awesome' . $min . '.css', '', '6.7.2');

	$fonts_url = travel_gem_fonts_url();

	if (! empty($fonts_url)) {
		wp_enqueue_style('travel-gem-google-fonts', $fonts_url, array(), null);
	}

	wp_enqueue_style('jquery-sidr', get_template_directory_uri() . '/third-party/sidr/css/jquery.sidr.dark' . $min . '.css', '', '2.2.1');

	wp_enqueue_style('travel-gem-style', get_stylesheet_uri(), null, date('Ymd-Gis', filemtime(get_template_directory() . '/style.css')));

	wp_enqueue_script('jquery-sidr', get_template_directory_uri() . '/third-party/sidr/js/jquery.sidr' . $min . '.js', array('jquery'), '2.2.1', true);

	wp_enqueue_script('travel-gem-navigation', get_template_directory_uri() . '/js/navigation.js', array(), '20151215', true);

	wp_enqueue_script('travel-gem-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20151215', true);

	wp_enqueue_script('travel-gem-custom', get_template_directory_uri() . '/js/custom' . $min . '.js', array('jquery', 'imagesloaded', 'masonry'), '1.0.0', true);

	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}

add_action('wp_enqueue_scripts', 'travel_gem_scripts');

if (! function_exists('travel_gem_blocks_support')) :
	/**
	 * Create add default blocks support
	 */
	function travel_gem_blocks_support()
	{
		// Add support for Block Styles.
		add_theme_support('wp-block-styles');

		// Add support for full and wide align images.
		add_theme_support('align-wide');

		// Add support for editor styles.
		add_theme_support('editor-styles');

		// Add support for responsive embeds.
		add_theme_support('responsive-embeds');

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__('Small', 'travel-gem'),
					'shortName' => esc_html__('S', 'travel-gem'),
					'size'      => 14,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__('Normal', 'travel-gem'),
					'shortName' => esc_html__('M', 'travel-gem'),
					'size'      => 18,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__('Large', 'travel-gem'),
					'shortName' => esc_html__('L', 'travel-gem'),
					'size'      => 42,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__('Huge', 'travel-gem'),
					'shortName' => esc_html__('XL', 'travel-gem'),
					'size'      => 54,
					'slug'      => 'huge',
				),
			)
		);

		// Add support for custom color scheme.
		add_theme_support('editor-color-palette', array(
			array(
				'name'  => esc_html__('White', 'travel-gem'),
				'slug'  => 'white',
				'color' => '#ffffff',
			),
			array(
				'name'  => esc_html__('Black', 'travel-gem'),
				'slug'  => 'black',
				'color' => '#111111',
			),
			array(
				'name'  => esc_html__('Gray', 'travel-gem'),
				'slug'  => 'gray',
				'color' => '#f4f4f4',
			),
			array(
				'name'  => esc_html__('Blue', 'travel-gem'),
				'slug'  => 'blue',
				'color' => '#21759b',
			),
			array(
				'name'  => esc_html__('Orange', 'travel-gem'),
				'slug'  => 'orange',
				'color' => '#FF7550',
			),
		));
	}
	add_action('after_setup_theme', 'travel_gem_blocks_support', 20);
endif; //travel_gem_blocks_support

if (! function_exists('travel_gem_add_blocks_style')) :
	/**
	 * Add Blocks Style
	 */
	function travel_gem_add_blocks_style()
	{
		// Theme block stylesheet.
		wp_enqueue_style('travel-gem-block-style', get_theme_file_uri('/css/blocks.css'), array('travel-gem-style'), date('Ymd-Gis', filemtime(get_template_directory() . '/css/blocks.css')));
	}
	add_action('wp_enqueue_scripts', 'travel_gem_add_blocks_style');
endif; //travel_gem_add_blocks_style

if (! function_exists('travel_gem_block_editor_styles')) :
	/**
	 * Enqueue editor styles for Blocks
	 */
	function travel_gem_block_editor_styles()
	{
		// Block styles.
		wp_enqueue_style('travel-gem-block-editor-style', get_theme_file_uri('/css/editor-blocks.css'), null, date('Ymd-Gis', filemtime(get_template_directory() . '/css/editor-blocks.css')));

		// Add custom fonts.
		wp_enqueue_style('travel-gem-fonts', travel_gem_fonts_url(), array(), null);
	}
	add_action('enqueue_block_editor_assets', 'travel_gem_block_editor_styles');
endif; //travel_gem_block_editor_styles

/**
 * Load init.
 */
require_once trailingslashit(get_template_directory()) . 'inc/init.php';
